// API URL - для прода измените в config.js
// ВАЖНО: Поиск конкурентов теперь идет напрямую к WB API с клиентского IP,
// чтобы снизить нагрузку на наш сервер и избежать rate limits
const API_URL = typeof CONFIG !== 'undefined' ? CONFIG.API_URL : 'http://localhost:8000';

let currentArticle = null;
let productData = null;
let authToken = null;

// Элементы DOM
const statusEl = document.getElementById('status');
const authRequiredEl = document.getElementById('auth-required');
const productInfoEl = document.getElementById('product-info');
const inputFormEl = document.getElementById('input-form');
const resultsEl = document.getElementById('results');
const errorEl = document.getElementById('error');

const loginBtn = document.getElementById('login-btn');
const logoutBtn = document.getElementById('logout-btn');
const calculateBtn = document.getElementById('calculate-btn');
const recalculateBtn = document.getElementById('recalculate-btn');
const costPriceInput = document.getElementById('cost-price');
const warehouseSelect = document.getElementById('warehouse');
const priceBeforeSppInput = document.getElementById('price-before-spp');
const competitorsSection = document.getElementById('competitors-section');
const competitorsSearchBtn = document.getElementById('competitors-search-btn');
const competitorsListEl = document.getElementById('competitors-list');
const competitorsStatsEl = document.getElementById('competitors-stats');
const competitorsPaginationEl = document.getElementById('competitors-pagination');
const competitorsPageInfoEl = document.getElementById('competitors-page-info');
const competitorsPrevBtn = document.getElementById('competitors-prev');
const competitorsNextBtn = document.getElementById('competitors-next');
const competitorsErrorEl = document.getElementById('competitors-error');
const competitorsEmptyEl = document.getElementById('competitors-empty');

const COMPETITORS_PAGE_SIZE = 5;
let competitorsState = {
  currentQuery: '',
  currentPage: 1,
  totalPages: 1,
  loading: false,
  allItems: [] // Хранит все загруженные конкуренты для пагинации
};
let competitorsAutoLoaded = false;

if (competitorsSearchBtn) {
  competitorsSearchBtn.addEventListener('click', () => loadCategoryCompetitors());
}
if (competitorsPrevBtn) {
  competitorsPrevBtn.addEventListener('click', () => {
    if (competitorsState.currentPage > 1) {
      // Локальная пагинация - не делаем новый запрос к API
      showCompetitorsPage(competitorsState.currentPage - 1);
    }
  });
}

if (competitorsNextBtn) {
  competitorsNextBtn.addEventListener('click', () => {
    if (competitorsState.currentPage < competitorsState.totalPages) {
      // Локальная пагинация - не делаем новый запрос к API
      showCompetitorsPage(competitorsState.currentPage + 1);
    }
  });
}

// Обработчик кнопки выхода
if (logoutBtn) {
  logoutBtn.addEventListener('click', () => {
    if (confirm('Вы уверены, что хотите выйти?')) {
      chrome.storage.local.remove(['wb_calc_token'], () => {
        console.log('🚪 User logged out');
        window.location.reload();
      });
    }
  });
}

// Обработчик кнопки входа
if (loginBtn) {
  loginBtn.addEventListener('click', () => {
    chrome.tabs.create({ url: chrome.runtime.getURL('login.html') });
  });
}

// Проверка авторизации при открытии
document.addEventListener('DOMContentLoaded', async () => {
  console.log('🔵 Extension loaded, checking auth...');
  
  // Получаем токен из chrome.storage
  chrome.storage.local.get(['wb_calc_token'], async (result) => {
    authToken = result.wb_calc_token;
    console.log('🔑 Token from storage:', authToken ? authToken.substring(0, 50) + '...' : 'NOT FOUND');
    
    if (!authToken) {
      console.log('❌ No token, showing login screen');
      statusEl.classList.add('hidden');
      authRequiredEl.classList.remove('hidden');
      return;
    }
    
    // Проверяем токен и подписку (всегда свежий запрос к API)
    try {
      console.log('📡 Verifying token with API:', `${API_URL}/auth/verify`);
      const verifyResponse = await fetch(`${API_URL}/auth/verify`, {
        headers: { 
          'Authorization': `Bearer ${authToken}`,
          'Cache-Control': 'no-cache'
        }
      }).catch(err => {
        console.error('❌ Network error:', err);
        throw new Error('Не удается подключиться к серверу. Проверьте что backend запущен на http://localhost:8000');
      });
      
      console.log('📥 API Response status:', verifyResponse.status);
      
      if (!verifyResponse.ok) {
        console.log('❌ Token invalid, clearing storage');
        // Токен недействителен - очищаем и показываем форму входа
        chrome.storage.local.remove(['wb_calc_token']);
        statusEl.classList.add('hidden');
        authRequiredEl.classList.remove('hidden');
        return;
      }
      
      const userData = await verifyResponse.json();
      console.log('👤 User data:', JSON.stringify(userData, null, 2));
      console.log('📅 subscription_until:', userData.subscription_until);
      console.log('✅ subscription_active:', userData.subscription_active);
      
      if (!userData.subscription_active) {
        console.log('⚠️ Subscription not active');
        showStatus('Подписка не активна. Оплатите доступ на https://your-landing.com', 'warning');
        return;
      }
      
      console.log('✅ All checks passed, initializing extension');
      // Всё ок, продолжаем работу
      logoutBtn.classList.remove('hidden'); // Показываем кнопку выхода
      await initializeExtension();
      
    } catch (error) {
      console.error('❌ Error during auth check:', error);
      showStatus('Ошибка соединения с сервером: ' + error.message, 'error');
    }
  });
});

// Инициализация расширения после успешной авторизации
async function initializeExtension() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (!tab.url || !tab.url.includes('wildberries.ru/catalog/') || !tab.url.includes('/detail.aspx')) {
      showStatus('Откройте страницу товара на Wildberries', 'warning');
      return;
    }
    
    // Извлекаем артикул прямо из URL
    const urlMatch = tab.url.match(/\/catalog\/(\d+)\//);
    if (urlMatch && urlMatch[1]) {
      currentArticle = urlMatch[1];
      console.log('📦 Article from URL:', currentArticle);
      await loadProduct(currentArticle);
    } else {
      showStatus('Не удалось определить артикул товара', 'error');
    }
    
  } catch (error) {
    console.error('Error:', error);
    showStatus('Ошибка инициализации: ' + error.message, 'error');
  }
}

// Загружаем список складов
async function loadWarehouses() {
  try {
    const response = await fetch(`${API_URL}/warehouses`);
    const data = await response.json();
    
    warehouseSelect.innerHTML = '';
    data.warehouses.forEach(wh => {
      const option = document.createElement('option');
      option.value = wh;
      option.textContent = wh;
      warehouseSelect.appendChild(option);
    });
    
    // Устанавливаем Коледино по умолчанию
    warehouseSelect.value = 'Коледино';
  } catch (error) {
    console.error('Failed to load warehouses:', error);
  }
}

// Загружаем информацию о товаре
async function loadProduct(article) {
  showStatus('Загрузка данных товара...', 'info');
  
  try {
    const response = await fetch(`${API_URL}/calculate`, {
      method: 'POST',
      headers: { 
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${authToken}`
      },
      body: JSON.stringify({ 
        article: parseInt(article), 
        cost_price: 0
      })
    });
    
    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.detail || 'Ошибка загрузки товара');
    }
    
    productData = await response.json();
    displayProduct(productData);
    
    await loadWarehouses();
    
    productInfoEl.classList.remove('hidden');
    inputFormEl.classList.remove('hidden');
    statusEl.classList.add('hidden');
    
    // Показываем секцию конкурентов после загрузки товара
    competitorsSection?.classList.remove('hidden');
    
  } catch (error) {
    showStatus('Ошибка: ' + error.message, 'error');
  }
}

// Отображаем информацию о товаре
function displayProduct(data) {
  document.getElementById('article').textContent = data.article;
  document.getElementById('name').textContent = data.name || 'Без названия';
  document.getElementById('category').textContent = data.category;
  document.getElementById('price').textContent = `${data.price_rub.toFixed(2)} ₽`;
  document.getElementById('volume').textContent = data.volume_liters ? `${data.volume_liters.toFixed(1)} л` : '—';
  document.getElementById('weight').textContent = data.weight_kg ? `${data.weight_kg.toFixed(2)} кг` : '—';
  
  // Отображаем фиксированные параметры
  document.getElementById('tax-info').textContent = `${(data.tax_rate * 100).toFixed(0)}%`;
  document.getElementById('defect-info').textContent = `${(data.defect_rate * 100).toFixed(0)}%`;
  
  // Процент выкупа (без отображения категории)
  const buyoutPercent = `${(data.buyout_rate * 100).toFixed(0)}%`;
  document.getElementById('buyout-info').textContent = buyoutPercent;

  // Секция конкурентов будет показана автоматически после расчёта
}

// Рассчитываем экономику
calculateBtn.addEventListener('click', async () => {
  const costPrice = parseFloat(costPriceInput.value);
  const priceBeforeSppValue = priceBeforeSppInput.value.trim();
  let priceBeforeSpp = null;
  if (priceBeforeSppValue !== '') {
    const parsedManualPrice = parseFloat(priceBeforeSppValue);
    if (!parsedManualPrice || parsedManualPrice <= 0) {
      showError('Введите корректную цену до СПП');
      return;
    }
    priceBeforeSpp = parsedManualPrice;
  }
  
  if (!costPrice || costPrice <= 0) {
    showError('Введите корректную себестоимость');
    return;
  }
  
  const useFbo = document.querySelector('input[name="mode"]:checked').value === 'fbo';
  const warehouse = warehouseSelect.value;
  
  calculateBtn.disabled = true;
  calculateBtn.textContent = 'Расчёт...';
  hideError();
  
  try {
    const response = await fetch(`${API_URL}/calculate`, {
      method: 'POST',
      headers: { 
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${authToken}`
      },
      body: JSON.stringify({
        article: currentArticle,
        cost_price: costPrice,
        use_fbo: useFbo,
        warehouse_name: warehouse,
        price_before_spp: priceBeforeSpp
      })
    });
    
    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.detail || 'Ошибка расчёта');
    }
    
    const result = await response.json();
    
    // Сохраняем конкурентов из ответа (но не показываем автоматически)
    if (result.competitors) {
      window.cachedCompetitors = result.competitors;
      console.log('📦 Competitors cached:', result.competitors.total_found, 'items');
    }
    
    displayResults(result);
    
    inputFormEl.classList.add('hidden');
    resultsEl.classList.remove('hidden');
    
  } catch (error) {
    showError('Ошибка: ' + error.message);
  } finally {
    calculateBtn.disabled = false;
    calculateBtn.textContent = 'Рассчитать';
  }
});

// Отображаем результаты
function displayResults(data) {
  document.getElementById('mode-label').textContent = data.mode;
  document.getElementById('commission').textContent = `${data.commission_rub.toFixed(2)} ₽`;
  document.getElementById('commission-rate').textContent = `${(data.commission_rate_used * 100).toFixed(1)}%`;
  document.getElementById('logistics').textContent = `${data.logistics_rub.toFixed(2)} ₽`;
  document.getElementById('logistics-return').textContent = `${data.logistics_with_return_rub.toFixed(2)} ₽`;
  document.getElementById('storage').textContent = `${data.storage_per_day_rub.toFixed(2)} ₽`;
  
  // Налог и брак
  document.getElementById('tax').textContent = `${data.tax_rub.toFixed(2)} ₽`;
  document.getElementById('tax-rate-display').textContent = `${(data.tax_rate * 100).toFixed(1)}%`;
  document.getElementById('defect').textContent = `${data.defect_rub.toFixed(2)} ₽`;
  document.getElementById('defect-rate-display').textContent = `${(data.defect_rate * 100).toFixed(1)}%`;
  
  // Процент выкупа
  const buyoutText = `${(data.buyout_rate * 100).toFixed(0)}%`;
  document.getElementById('buyout-rate-display').textContent = buyoutText;
  
  const profit = data.gross_profit_rub;
  const profitEl = document.getElementById('profit');
  profitEl.textContent = `${profit.toFixed(2)} ₽`;
  profitEl.className = 'value ' + (profit > 0 ? 'positive' : profit < 0 ? 'negative' : '');
  
  document.getElementById('margin').textContent = `${data.profitability_percent.toFixed(1)}%`;
  document.getElementById('minimal-price').textContent = `${data.minimal_price_rub.toFixed(2)} ₽`;
}

// Пересчитать
recalculateBtn.addEventListener('click', () => {
  resultsEl.classList.add('hidden');
  inputFormEl.classList.remove('hidden');
  costPriceInput.focus();
});

// ============== НОВЫЙ МЕТОД: Загрузка похожих товаров через WB internal API ==============

const COMPETITORS_CACHE_TTL = 5 * 60 * 1000; // 5 минут в миллисекундах

async function loadCategoryCompetitors(pageOverride) {
  if (!currentArticle) {
    showCompetitorsError('Артикул товара не определён');
    return;
  }

  const page = pageOverride ?? 1;
  const cacheKey = `competitors_${currentArticle}`;
  
  // Проверяем кэш
  try {
    const cached = await new Promise(resolve => {
      chrome.storage.local.get([cacheKey], result => resolve(result[cacheKey]));
    });
    
    if (cached && cached.timestamp && (Date.now() - cached.timestamp < COMPETITORS_CACHE_TTL)) {
      console.log('[Competitors] Загружено из кэша, возраст:', Math.round((Date.now() - cached.timestamp) / 1000), 'сек');
      
      const items = cached.items || [];
      const startIdx = (page - 1) * COMPETITORS_PAGE_SIZE;
      const endIdx = startIdx + COMPETITORS_PAGE_SIZE;
      const pageItems = items.slice(startIdx, endIdx);
      const totalPages = Math.ceil(items.length / COMPETITORS_PAGE_SIZE);
      
      competitorsState = {
        currentQuery: `similar_${currentArticle}`,
        currentPage: page,
        totalPages: totalPages,
        loading: false,
        allItems: items
      };
      
      competitorsSection?.classList.remove('hidden');
      renderCompetitors({
        total_found: cached.total_found,
        showing: items.length,
        page: page,
        page_size: COMPETITORS_PAGE_SIZE,
        total_pages: totalPages,
        items: pageItems,
        cached: true,
        query: `Похожие на ${currentArticle}`
      });
      return;
    }
  } catch (e) {
    console.warn('[Competitors] Ошибка чтения кэша:', e);
  }

  setCompetitorsLoading(true);
  clearCompetitorsError();
  hideCompetitorsEmptyState();
  competitorsSection?.classList.remove('hidden');

  try {
    // Добавляем небольшую задержку чтобы не выглядеть как бот
    await new Promise(resolve => setTimeout(resolve, 300 + Math.random() * 200));
    
    // Используем внутренний API WB для получения похожих товаров
    // Этот запрос идёт напрямую из браузера пользователя с его cookies
    const wbRecomUrl = 'https://www.wildberries.ru/__internal/recom/recom/ru/common/v8/search';
    const params = new URLSearchParams({
      query: `похожие ${currentArticle}`,
      resultset: 'catalog',
      page: 1,
      appType: 1,
      curr: 'rub',
      dest: -1257786,
      spp: 30,
      lang: 'ru',
      ab_testing: 'false',
      suppressSpellcheck: 'false'
    });

    console.log('[Competitors] Запрос к WB Recom API:', `${wbRecomUrl}?${params}`);
    
    const response = await fetch(`${wbRecomUrl}?${params}`, {
      method: 'GET',
      credentials: 'include', // Важно: отправляем cookies пользователя
      headers: {
        'Accept': 'application/json',
        'Accept-Language': 'ru-RU,ru;q=0.9,en-US;q=0.8,en;q=0.7',
        'User-Agent': navigator.userAgent,
        'Referer': 'https://www.wildberries.ru/',
        'Origin': 'https://www.wildberries.ru',
        'sec-ch-ua': '"Google Chrome";v="131", "Chromium";v="131", "Not_A Brand";v="24"',
        'sec-ch-ua-mobile': '?0',
        'sec-ch-ua-platform': '"Windows"',
        'sec-fetch-dest': 'empty',
        'sec-fetch-mode': 'cors',
        'sec-fetch-site': 'same-origin'
      }
    });

    if (!response.ok) {
      if (response.status === 429) {
        throw new Error('WB временно ограничил запросы. Подождите минуту.');
      }
      throw new Error(`WB API error: ${response.status}`);
    }

    const wbData = await response.json();
    const products = wbData?.products || [];
    
    console.log(`[Competitors] Получено ${products.length} похожих товаров`);
    
    // Фильтруем - исключаем текущий товар и товары того же продавца
    const currentSupplier = productData?.supplierId || null;
    
    const items = products
      .filter(product => {
        // Исключаем текущий товар
        if (product.id === currentArticle) return false;
        // Опционально: исключаем товары того же продавца
        // if (currentSupplier && product.supplierId === currentSupplier) return false;
        return true;
      })
      .slice(0, 20) // Берём топ 20
      .map(product => {
        const article = product.id;
        const name = product.name || '';
        const brand = product.brand || '';
        
        // Цена
        let price = 0;
        if (product.sizes && product.sizes.length > 0) {
          const priceData = product.sizes[0].price;
          if (priceData && typeof priceData === 'object') {
            price = (priceData.product || 0) / 100;
          }
        }
        
        // Рейтинг и отзывы
        const rating = product.reviewRating || 0;
        const reviews = product.feedbacks || 0;
        
        // Продавец
        const supplier = product.supplier || '';
        
        // Изображение - WB basket формула
        let image_url = null;
        if (article) {
          const vol = Math.floor(article / 100000);
          const part = Math.floor(article / 1000);
          
          // Определяем номер basket на основе vol (официальная логика WB)
          let basket;
          if (vol >= 0 && vol <= 143) basket = '01';
          else if (vol >= 144 && vol <= 287) basket = '02';
          else if (vol >= 288 && vol <= 431) basket = '03';
          else if (vol >= 432 && vol <= 719) basket = '04';
          else if (vol >= 720 && vol <= 1007) basket = '05';
          else if (vol >= 1008 && vol <= 1061) basket = '06';
          else if (vol >= 1062 && vol <= 1115) basket = '07';
          else if (vol >= 1116 && vol <= 1169) basket = '08';
          else if (vol >= 1170 && vol <= 1313) basket = '09';
          else if (vol >= 1314 && vol <= 1601) basket = '10';
          else if (vol >= 1602 && vol <= 1655) basket = '11';
          else if (vol >= 1656 && vol <= 1919) basket = '12';
          else if (vol >= 1920 && vol <= 2045) basket = '13';
          else if (vol >= 2046 && vol <= 2189) basket = '14';
          else if (vol >= 2190 && vol <= 2405) basket = '15';
          else if (vol >= 2406 && vol <= 2621) basket = '16';
          else if (vol >= 2622 && vol <= 2837) basket = '17';
          else if (vol >= 2838 && vol <= 3053) basket = '18';
          else if (vol >= 3054 && vol <= 3269) basket = '19';
          else if (vol >= 3270 && vol <= 3485) basket = '20';
          else if (vol >= 3486 && vol <= 3701) basket = '21';
          else if (vol >= 3702 && vol <= 3917) basket = '22';
          else basket = '23'; // для vol >= 3918
          
          image_url = `https://basket-${basket}.wbbasket.ru/vol${vol}/part${part}/${article}/images/c246x328/1.webp`;
        }
        
        // Ссылка на товар
        const product_url = `https://www.wildberries.ru/catalog/${article}/detail.aspx`;
        
        return {
          article,
          name,
          brand,
          price,
          rating,
          reviews,
          supplier,
          image_url,
          product_url
        };
      });
    
    // Сохраняем в кэш chrome.storage.local
    const cacheData = {
      total_found: wbData.total || items.length,
      items: items,
      timestamp: Date.now()
    };
    
    chrome.storage.local.set({ [cacheKey]: cacheData }, () => {
      console.log('[Competitors] Сохранено в кэш на 5 минут');
    });
    
    // Пагинация локально (по 5 на страницу)
    const startIdx = (page - 1) * COMPETITORS_PAGE_SIZE;
    const endIdx = startIdx + COMPETITORS_PAGE_SIZE;
    const pageItems = items.slice(startIdx, endIdx);
    const totalPages = Math.ceil(items.length / COMPETITORS_PAGE_SIZE);
    
    const data = {
      total_found: wbData.total || items.length,
      showing: items.length,
      page: page,
      page_size: COMPETITORS_PAGE_SIZE,
      total_pages: totalPages,
      items: pageItems,
      cached: false,
      query: `Похожие на ${currentArticle}`
    };
    
    competitorsState = {
      currentQuery: `similar_${currentArticle}`,
      currentPage: page,
      totalPages: totalPages,
      loading: false,
      allItems: items // Сохраняем все элементы для пагинации
    };
    
    renderCompetitors(data);

  } catch (error) {
    console.error('Competitors search error:', error);
    showCompetitorsError(error.message || 'Ошибка загрузки конкурентов');
  } finally {
    setCompetitorsLoading(false);
  }
}

// Старая функция поиска - теперь вызывает loadCategoryCompetitors
async function performCompetitorsSearch(pageOverride) {
  return loadCategoryCompetitors(pageOverride);
}

function showCompetitorsPage(page) {
  // Показываем страницу из уже загруженных данных (без нового запроса)
  if (!competitorsState.allItems || competitorsState.allItems.length === 0) {
    return;
  }
  
  const startIdx = (page - 1) * COMPETITORS_PAGE_SIZE;
  const endIdx = startIdx + COMPETITORS_PAGE_SIZE;
  const pageItems = competitorsState.allItems.slice(startIdx, endIdx);
  
  const data = {
    total_found: competitorsState.allItems.length,
    showing: competitorsState.allItems.length,
    page: page,
    page_size: COMPETITORS_PAGE_SIZE,
    total_pages: competitorsState.totalPages,
    items: pageItems,
    cached: true,
    query: competitorsState.currentQuery
  };
  
  competitorsState.currentPage = page;
  renderCompetitors(data);
}

function renderCompetitors(data) {
  if (!competitorsListEl) return;

  updateCompetitorsStats(data);

  if (!Array.isArray(data.items) || data.items.length === 0) {
    competitorsListEl.innerHTML = '';
    showCompetitorsEmptyState('Для этого запроса конкуренты не найдены.');
    updateCompetitorsPagination({ page: 1, total_pages: 1 });
    return;
  }

  hideCompetitorsEmptyState();

  const itemsHtml = data.items.map((item) => {
    const priceText = formatPrice(item.price);
    const ratingText = typeof item.rating === 'number' && item.rating > 0 ? item.rating.toFixed(1) : '—';
    const reviewsText = typeof item.reviews === 'number' ? item.reviews : 0;
    const imageSrc = item.image_url || 'https://via.placeholder.com/60x80?text=WB';
    const productUrl = item.product_url || '#';

    return `
      <div class="competitor-item">
        <img 
          src="${imageSrc}" 
          alt="${item.name || 'Товар'}" 
          class="competitor-thumb" 
          loading="lazy"
          onerror="this.onerror=null; this.src='https://via.placeholder.com/60x80/6366f1/ffffff?text=WB';"
        >
        <div class="competitor-text">
          <div class="competitor-brand">${item.brand || '—'}</div>
          <div class="competitor-name" title="${item.name || ''}">${item.name || 'Без названия'}</div>
          <div class="competitor-meta">
            <span class="price">${priceText}</span>
            <span>⭐ ${ratingText}</span>
            <span>(${reviewsText})</span>
          </div>
        </div>
        <a href="${productUrl}" target="_blank" rel="noopener noreferrer" class="competitor-link">Открыть →</a>
      </div>
    `;
  }).join('');

  competitorsListEl.innerHTML = itemsHtml;
  updateCompetitorsPagination(data);
}

function updateCompetitorsStats(data) {
  if (!competitorsStatsEl) return;
  const totalFound = data.total_found || 0;
  const showing = data.showing || (Array.isArray(data.items) ? data.items.length : 0);
  const pageSize = data.page_size || COMPETITORS_PAGE_SIZE;

  if (!totalFound) {
    competitorsStatsEl.textContent = 'Похожие товары от других продавцов.';
    return;
  }

  competitorsStatsEl.textContent = `Найдено: ${formatNumber(totalFound)} • показываем ТОП ${formatNumber(showing)} (по ${pageSize} на страницу)`;
}

function updateCompetitorsPagination(data) {
  if (!competitorsPaginationEl || !competitorsPageInfoEl) return;
  const totalPages = Math.max(1, data.total_pages || 1);
  const currentPage = Math.min(totalPages, data.page || 1);

  competitorsPageInfoEl.textContent = `${currentPage} / ${totalPages}`;

  if (competitorsPrevBtn) {
    competitorsPrevBtn.disabled = currentPage <= 1;
  }
  if (competitorsNextBtn) {
    competitorsNextBtn.disabled = currentPage >= totalPages;
  }

  if (totalPages <= 1) {
    competitorsPaginationEl.classList.add('hidden');
  } else {
    competitorsPaginationEl.classList.remove('hidden');
  }
}

function setCompetitorsLoading(isLoading) {
  if (!competitorsSearchBtn) return;
  competitorsSearchBtn.disabled = isLoading;
  competitorsSearchBtn.textContent = isLoading ? 'Ищем...' : 'Искать';
}

function showCompetitorsError(message) {
  if (!competitorsErrorEl) return;
  competitorsErrorEl.textContent = message;
  competitorsErrorEl.classList.remove('hidden');
}

function clearCompetitorsError() {
  if (!competitorsErrorEl) return;
  competitorsErrorEl.classList.add('hidden');
}

function showCompetitorsEmptyState(message) {
  if (!competitorsEmptyEl) return;
  competitorsEmptyEl.textContent = message;
  competitorsEmptyEl.classList.remove('hidden');
}

function hideCompetitorsEmptyState() {
  if (!competitorsEmptyEl) return;
  competitorsEmptyEl.classList.add('hidden');
}

function formatPrice(value) {
  if (typeof value !== 'number' || Number.isNaN(value) || value <= 0) {
    return '—';
  }
  return `${value.toLocaleString('ru-RU', { maximumFractionDigits: 0 })} ₽`;
}

function formatNumber(value) {
  if (typeof value !== 'number' || Number.isNaN(value)) {
    return '0';
  }
  return value.toLocaleString('ru-RU');
}

// Показать статус
function showStatus(message, type = 'info') {
  statusEl.textContent = message;
  statusEl.className = `status ${type}`;
  statusEl.classList.remove('hidden');
}

// Показать/скрыть ошибку
function showError(message) {
  errorEl.textContent = message;
  errorEl.classList.remove('hidden');
}

function hideError() {
  errorEl.classList.add('hidden');
}
