// API URL - для прода измените в config.js
const API_URL = typeof CONFIG !== 'undefined' ? CONFIG.API_URL : 'http://localhost:8000';
const LANDING_URL = typeof CONFIG !== 'undefined' ? CONFIG.LANDING_URL : 'http://127.0.0.1:5565';

// Rate limiting - защита от брутфорса на клиенте
const RateLimiter = {
  attempts: 0,
  lastAttempt: 0,
  maxAttempts: 5,
  lockoutTime: 60000, // 1 минута
  
  canAttempt() {
    const now = Date.now();
    // Сброс после lockout периода
    if (now - this.lastAttempt > this.lockoutTime) {
      this.attempts = 0;
    }
    return this.attempts < this.maxAttempts;
  },
  
  recordAttempt() {
    this.attempts++;
    this.lastAttempt = Date.now();
  },
  
  getRemainingTime() {
    const elapsed = Date.now() - this.lastAttempt;
    return Math.ceil((this.lockoutTime - elapsed) / 1000);
  },
  
  reset() {
    this.attempts = 0;
  }
};

// Input sanitization - защита от XSS
function sanitizeInput(str) {
  if (typeof str !== 'string') return '';
  return str
    .trim()
    .replace(/[<>]/g, '') // Удаляем потенциальные HTML теги
    .substring(0, 255);   // Ограничиваем длину
}

// Email validation
function isValidEmail(email) {
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  return emailRegex.test(email) && email.length <= 255;
}

// DOM Elements
const form = document.getElementById('loginForm');
const loginBtn = document.getElementById('loginBtn');
const errorMessage = document.getElementById('errorMessage');
const successMessage = document.getElementById('successMessage');
const togglePassword = document.getElementById('togglePassword');
const passwordInput = document.getElementById('password');
const registerLink = document.getElementById('registerLink');

// Set registration link
if (registerLink) {
  registerLink.href = LANDING_URL;
}

// Password visibility toggle
if (togglePassword && passwordInput) {
  togglePassword.addEventListener('click', () => {
    const type = passwordInput.type === 'password' ? 'text' : 'password';
    passwordInput.type = type;
    togglePassword.textContent = type === 'password' ? '👁️' : '🙈';
    // Update aria-label based on language
    if (typeof t === 'function') {
      togglePassword.setAttribute('aria-label', t(type === 'password' ? 'login.showPassword' : 'login.hidePassword'));
    }
  });
}

// Show error with sanitization (uses i18n if available)
function showError(message) {
  const safeMessage = sanitizeInput(message);
  errorMessage.textContent = safeMessage;
  errorMessage.classList.add('show');
  successMessage.classList.remove('show');
}

// Show success
function showSuccess(message) {
  successMessage.textContent = message;
  successMessage.classList.add('show');
  errorMessage.classList.remove('show');
}

// Reset button state (uses i18n)
function resetButton() {
  loginBtn.disabled = false;
  loginBtn.innerHTML = typeof t === 'function' ? t('login.btn') : 'Войти';
}

// Set loading state (uses i18n)
function setLoading() {
  loginBtn.disabled = true;
  const loadingText = typeof t === 'function' ? t('login.btnLoading') : 'Вход...';
  loginBtn.innerHTML = `<span class="spinner"></span>${loadingText}`;
}

form.addEventListener('submit', async (event) => {
  event.preventDefault();
  
  // Rate limiting check
  if (!RateLimiter.canAttempt()) {
    const msg = typeof t === 'function' 
      ? t('login.errorRateLimit', { seconds: RateLimiter.getRemainingTime() })
      : `Слишком много попыток. Подождите ${RateLimiter.getRemainingTime()} сек.`;
    showError(msg);
    return;
  }
  
  // Get and sanitize inputs
  const email = sanitizeInput(document.getElementById('email').value);
  const password = document.getElementById('password').value; // Don't sanitize password (can have special chars)
  
  // Validation
  if (!email || !password) {
    showError(typeof t === 'function' ? t('login.errorEmpty') : 'Заполните все поля');
    return;
  }
  
  if (!isValidEmail(email)) {
    showError(typeof t === 'function' ? t('login.errorEmail') : 'Введите корректный email');
    return;
  }
  
  // Убрана проверка минимальной длины пароля - 
  // старые пользователи могут иметь короткие пароли
  // Проверка длины делается на бэкенде при регистрации
  
  if (password.length > 128) {
    showError(typeof t === 'function' ? t('login.errorPasswordLong') : 'Пароль слишком длинный');
    return;
  }
  
  // Hide messages and set loading
  errorMessage.classList.remove('show');
  successMessage.classList.remove('show');
  setLoading();
  
  // Record attempt for rate limiting
  RateLimiter.recordAttempt();
  
  try {
    // Add timeout to fetch
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 15000); // 15 sec timeout
    
    const response = await fetch(`${API_URL}/auth/extension-login`, {
      method: 'POST',
      headers: { 
        'Content-Type': 'application/json',
        'X-Requested-With': 'XMLHttpRequest' // CSRF protection hint
      },
      body: JSON.stringify({ email, password }),
      signal: controller.signal
    });
    
    clearTimeout(timeoutId);
    
    if (!response.ok) {
      let errorText = typeof t === 'function' ? t('login.errorLogin') : 'Ошибка входа';
      try {
        const error = await response.json();
        errorText = error.detail || errorText;
      } catch {
        // Response wasn't JSON
      }
      throw new Error(errorText);
    }
    
    const data = await response.json();
    
    // Validate response has token
    if (!data.access_token || typeof data.access_token !== 'string') {
      throw new Error(typeof t === 'function' ? t('login.errorServer') : 'Некорректный ответ сервера');
    }
    
    // Reset rate limiter on success
    RateLimiter.reset();
    
    // Сохраняем токен в chrome.storage (secure storage)
    chrome.storage.local.set({ 
      wb_calc_token: data.access_token,
      wb_calc_token_time: Date.now() // For potential token expiry check
    }, () => {
      showSuccess(typeof t === 'function' ? t('login.success') : 'Вход выполнен успешно! Можете закрыть эту вкладку.');
      
      // Закрываем вкладку через 2 секунды
      setTimeout(() => {
        window.close();
      }, 2000);
    });
    
  } catch (error) {
    if (error.name === 'AbortError') {
      showError(typeof t === 'function' ? t('login.errorTimeout') : 'Превышено время ожидания. Проверьте соединение.');
    } else {
      showError(error.message || (typeof t === 'function' ? t('login.errorGeneric') : 'Произошла ошибка'));
    }
    resetButton();
  }
});

// Prevent paste in password field (optional, for extra security)
// Uncomment if needed:
// passwordInput.addEventListener('paste', (e) => e.preventDefault());
